﻿/*
  KeeUIExt
  Copyright (C) 2014-2025 Dominik Reichl <dominik.reichl@t-online.de>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Reflection;
using System.Text;
using System.Windows.Forms;

using KeePass.App;
using KeePass.App.Configuration;
using KeePass.Plugins;
using KeePass.UI;

using KeePassLib.Utility;

namespace KeeUIExt.Forms
{
	public partial class KueOptionsForm : Form
	{
		private CheckedLVItemDXList m_cdx = null;
		private object m_fcgUI = null;

		[Browsable(false)]
		[DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
		[DefaultValue((object)null)]
		internal KueOptions KueOptions { get; set; }

		[Browsable(false)]
		[DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
		[DefaultValue((object)null)]
		internal IPluginHost PluginHost { get; set; }

		public KueOptionsForm()
		{
			InitializeComponent();
		}

		private void OnFormLoad(object sender, EventArgs e)
		{
			KueOptions opt = this.KueOptions;
			if(opt == null)
			{
				Debug.Assert(false);
				opt = new KueOptions();
				this.KueOptions = opt;
			}

			GlobalWindowManager.AddWindow(this);

			string strTitle = KeeUIExtExt.ProductName + " Options";
			this.Text = strTitle;
			this.Icon = AppIcons.Default;

			BannerFactory.CreateBannerEx(this, m_bannerImage, ((this.PluginHost != null) ?
				(this.PluginHost.Resources.GetObject("B48x48_KCMSystem") as Image) : null),
				strTitle, "Configure the " + KeeUIExtExt.ProductName + " plugin.");

			UIUtil.SetExplorerTheme(m_lvOptions, false);
			m_lvOptions.Columns.Add(string.Empty, m_lvOptions.ClientSize.Width -
				UIUtil.GetVScrollBarWidth());

			m_cdx = new CheckedLVItemDXList(m_lvOptions, false);

			ListViewGroup lvg = new ListViewGroup("General");
			m_lvOptions.Groups.Add(lvg);

			m_cdx.CreateItem(opt, "CtrlBackspace", lvg,
				"When pressing Ctrl+Backspace, delete the word to the left of the cursor");
			m_cdx.CreateItem(opt, "ScrollMenusWithMouseWheel", lvg,
				"Scroll menus with mouse wheel");

			lvg = new ListViewGroup("Main Window");
			m_lvOptions.Groups.Add(lvg);

			m_cdx.CreateItem(opt, "ForgetLastDatabaseIfNotExists", lvg,
				"Forget last used database if it does not exist anymore");
			m_cdx.CreateItem(opt, "ConfirmDatabaseClosing", lvg,
				"Show confirmation dialog when closing a database");

			lvg = new ListViewGroup("Main Group Tree");
			m_lvOptions.Groups.Add(lvg);

			m_cdx.CreateItem(opt, "MainGroupCollapseOnOpen", lvg,
				"Collapse all groups when opening a database");
			m_cdx.CreateItem(opt, "MainGroupContextDuplicate", lvg,
				"Show 'Duplicate Group' command in the group context menu *");

			lvg = new ListViewGroup("Main Entry List");
			m_lvOptions.Groups.Add(lvg);

			m_cdx.CreateItem(opt, "MainEntryFocusOnOpen", lvg,
				"Focus entry list when opening a database");
			m_cdx.CreateItem(opt, "MainEntryGridLines", lvg,
				"Draw grid lines");
			m_cdx.CreateItem(opt, "MainEntryContextDuplicate", lvg,
				"Show 'Duplicate Entry' command in the entry context menu *");
			m_cdx.CreateItem(opt, "MainEntryContextPrint", lvg,
				"Show 'Print Entry' command in the entry context menu *");

			m_cdx.UpdateData(false);

			try
			{
				Type tFcg = typeof(UIUtil).Assembly.GetType(
					"KeePass.UI.FontControlGroup", true);
				object[] vParams = new object[] { m_cbUIFont, m_btnUIFont,
					opt.UIFont, null };
				m_fcgUI = Activator.CreateInstance(tFcg, BindingFlags.Public |
					BindingFlags.NonPublic | BindingFlags.Instance |
					BindingFlags.CreateInstance, null, vParams, null);
			}
			catch(Exception)
			{
				Debug.Assert(false);
				m_cbUIFont.Enabled = false;
				m_btnUIFont.Enabled = false;
			}
		}

		private void OnFormClosed(object sender, FormClosedEventArgs e)
		{
			IDisposable d = (m_fcgUI as IDisposable);
			if(d != null) d.Dispose();
			else { Debug.Assert(false); }
			m_fcgUI = null;

			GlobalWindowManager.RemoveWindow(this);
		}

		private void OnBtnOK(object sender, EventArgs e)
		{
			m_cdx.UpdateData(true);

			if(m_fcgUI != null)
			{
				PropertyInfo pi = m_fcgUI.GetType().GetProperty("SelectedFont",
					BindingFlags.Public | BindingFlags.NonPublic | BindingFlags.Instance);
				if((pi != null) && (pi.PropertyType == typeof(AceFont)))
				{
					AceFont af = (AceFont)pi.GetValue(m_fcgUI, null);
					this.KueOptions.UIFont = (((af != null) && af.OverrideUIDefault) ?
						af : null);
				}
				else { Debug.Assert(false); }
			}
			else { Debug.Assert(false); }
		}

		private void OnLinkUIFontSize(object sender, LinkLabelLinkClickedEventArgs e)
		{
			AppHelp.ShowHelp("base/faq_tech", "guifont");
		}
	}
}
