﻿/*
  KeeUIExt
  Copyright (C) 2014-2025 Dominik Reichl <dominik.reichl@t-online.de>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Reflection;
using System.Text;
using System.Threading;
using System.Windows.Forms;

using KeePass;
using KeePass.App.Configuration;
using KeePass.Forms;
using KeePass.Plugins;
using KeePass.Resources;
using KeePass.UI;

using KeePassLib;
using KeePassLib.Delegates;
using KeePassLib.Native;
using KeePassLib.Serialization;
using KeePassLib.Utility;

using KeeUIExt.Forms;

namespace KeeUIExt
{
	public sealed class KeeUIExtExt : Plugin
	{
		internal const string ProductName = "KeeUIExt";

		private IPluginHost m_host = null;
		private KueOptions m_opt = null;

		private KueMessageFilter m_msgf = null;
		private object m_oMenuItemLinks = null;

		public override bool Initialize(IPluginHost host)
		{
			Terminate();
			if(host == null) { Debug.Assert(false); return false; }
			m_host = host;

			m_opt = KueOptions.Load(m_host.CustomConfig);

			GlobalWindowManager.WindowAdded += this.OnWindowAdded;
			GlobalWindowManager.WindowRemoved += this.OnWindowRemoved;
			m_host.MainWindow.FileOpened += this.OnFileOpened;
			m_host.MainWindow.FileClosingPre += this.OnFileClosingPre;

			m_msgf = new KueMessageFilter(m_opt);
			Application.AddMessageFilter(m_msgf);

			try { OnInitializeEx(); }
			catch(Exception) { Debug.Assert(false); }

			ApplyOptions();
			return true;
		}

		public override void Terminate()
		{
			if(m_host == null) return;

			Application.RemoveMessageFilter(m_msgf);
			m_msgf = null;

			GlobalWindowManager.WindowAdded -= this.OnWindowAdded;
			GlobalWindowManager.WindowRemoved -= this.OnWindowRemoved;
			m_host.MainWindow.FileOpened -= this.OnFileOpened;
			m_host.MainWindow.FileClosingPre -= this.OnFileClosingPre;

			KueOptions.Save(m_host.CustomConfig, m_opt);

			m_host = null;
		}

		public override ToolStripMenuItem GetMenuItem(PluginMenuType t)
		{
			if(t != PluginMenuType.Main) return null;

			ToolStripMenuItem tsmi = new ToolStripMenuItem();
			tsmi.Text = KeeUIExtExt.ProductName + " Options...";
			tsmi.Image = (m_host.Resources.GetObject("B16x16_Misc") as Image);
			tsmi.Click += this.OnOptions;

			return tsmi;
		}

		private void OnOptions(object sender, EventArgs e)
		{
			KueOptionsForm f = new KueOptionsForm();
			f.KueOptions = m_opt;
			f.PluginHost = m_host;

			if(UIUtil.ShowDialogAndDestroy(f) == DialogResult.OK)
				ApplyOptions();
		}

		private void OnInitializeEx()
		{
			MainForm mf = m_host.MainWindow;
			MenuStrip menu = mf.MainMenu;
			ContextMenuStrip ctxG = mf.GroupContextMenu;
			ContextMenuStrip ctxE = mf.EntryContextMenu;

			if(m_opt.ForgetLastDatabaseIfNotExists)
			{
				AceApplication ace = Program.Config.Application;
				IOConnectionInfo ioc = ace.LastUsedFile;
				if((ioc != null) && !string.IsNullOrEmpty(ioc.Path) &&
					!ioc.CanProbablyAccess())
					ace.LastUsedFile = new IOConnectionInfo();
			}

			if(m_opt.MainGroupContextDuplicate)
				CreateMenuItemCopy(ctxG.Items, FindMenuItem(ctxG, "m_ctxGroupEdit"),
					true, FindMenuItem(menu, "m_menuGroupDuplicate"));
			if(m_opt.MainEntryContextDuplicate)
				CreateMenuItemCopy(ctxE.Items, FindMenuItem(ctxE, "m_ctxEntryEditQuick"),
					true, FindMenuItem(menu, "m_menuEntryDuplicate"));
			if(m_opt.MainEntryContextPrint)
				CreateMenuItemCopy(ctxE.Items, null,
					true, FindMenuItem(menu, "m_menuEntryPrint"));
		}

		private void ApplyOptions()
		{
			CustomizeForm(m_host.MainWindow);

			ListView lv = FindEntryListView();
			if(lv != null) lv.GridLines = m_opt.MainEntryGridLines;
		}

		private static ToolStripMenuItem FindMenuItem(ToolStrip ts, string strName)
		{
			if(ts == null) { Debug.Assert(false); return null; }
			if(string.IsNullOrEmpty(strName)) { Debug.Assert(false); return null; }

			// Mono doesn't implement ToolStripItemCollection.Find properly
			if(MonoWorkarounds.IsRequired())
			{
				List<ToolStripItem> l = new List<ToolStripItem>();
				foreach(ToolStripItem tsi in ts.Items)
					FindToolStripItems(tsi, strName, l);
				if(l.Count != 1) return null;
				return (l[0] as ToolStripMenuItem);
			}

			ToolStripItem[] v = ts.Items.Find(strName, true);
			if((v == null) || (v.Length != 1)) return null;
			return (v[0] as ToolStripMenuItem);
		}

		private static void FindToolStripItems(ToolStripItem tsi, string strName,
			List<ToolStripItem> lFound)
		{
			if(tsi == null) { Debug.Assert(false); return; }
			if(string.IsNullOrEmpty(strName)) { Debug.Assert(false); return; }
			if(lFound == null) { Debug.Assert(false); return; }

			if(string.Equals(tsi.Name, strName, StringComparison.InvariantCultureIgnoreCase))
				lFound.Add(tsi);

			ToolStripDropDownItem tsddi = (tsi as ToolStripDropDownItem);
			if(tsddi != null)
			{
				foreach(ToolStripItem tsiSub in tsddi.DropDownItems)
					FindToolStripItems(tsiSub, strName, lFound);
			}
		}

		private static T FindControl<T>(Control cRoot, string strName)
			where T : Control
		{
			if(cRoot == null) { Debug.Assert(false); return null; }
			if(string.IsNullOrEmpty(strName)) { Debug.Assert(false); return null; }

			Control[] v = cRoot.Controls.Find(strName, true);
			if((v == null) || (v.Length != 1)) return null;
			return (v[0] as T);
		}

		private ListView FindEntryListView()
		{
			ListView lv = FindControl<ListView>(m_host.MainWindow, "m_lvEntries");
			Debug.Assert(lv != null);
			return lv;
		}

		private void CreateMenuItemCopy(ToolStripItemCollection tsicTarget,
			ToolStripItem tsiPosRef, bool bAfter, ToolStripMenuItem tsmiBase)
		{
			if(tsicTarget == null) { Debug.Assert(false); return; }
			if(tsmiBase == null) { Debug.Assert(false); return; }

			try
			{
				Type t = typeof(UIUtil).Assembly.GetType("KeePass.UI.MenuItemLinks");

				object o = m_oMenuItemLinks;
				if(o == null)
				{
					o = Activator.CreateInstance(t, true);
					m_oMenuItemLinks = o;
				}

				MethodInfo mi = t.GetMethod("CreateCopy", (BindingFlags.Instance |
					BindingFlags.NonPublic | BindingFlags.Public));
				mi.Invoke(o, new object[] { tsicTarget, tsiPosRef, bAfter, tsmiBase });
			}
			catch(Exception) { Debug.Assert(false); }
		}

		private void OnWindowAdded(object sender, GwmWindowEventArgs e)
		{
			Form f = ((e != null) ? e.Form : null);
			if(f == null) { Debug.Assert(false); return; }

			f.Shown += this.OnFormShown;

			CustomizeForm(f);
		}

		private void OnWindowRemoved(object sender, GwmWindowEventArgs e)
		{
			Form f = ((e != null) ? e.Form : null);
			if(f == null) { Debug.Assert(false); return; }

			f.Shown -= this.OnFormShown;
		}

		private void OnFormShown(object sender, EventArgs e)
		{
			Form f = (sender as Form);
			if(f == null) { Debug.Assert(false); return; }

			CustomizeControlOnShown(f);
		}

		private void CustomizeForm(Form f)
		{
			if(f == null) { Debug.Assert(false); return; }

			try
			{
				AceFont af = m_opt.UIFont;
				if((af != null) && af.OverrideUIDefault)
				{
					if(NativeLib.IsUnix())
						Program.Config.UI.ForceSystemFontUnix = false;

					Rectangle rOrg = f.Bounds;
					Font font = af.ToFont();

					FieldInfo fi = typeof(ToolStripManager).GetField("defaultFont",
						BindingFlags.NonPublic | BindingFlags.Static);
					if((fi != null) && (fi.FieldType == typeof(Font)))
						fi.SetValue(null, font);
					else { Debug.Assert(NativeLib.IsUnix()); }

					f.Font = font;

					MainForm mf = (f as MainForm);
					if(mf != null)
					{
						Action<Control> fC = delegate(Control c)
						{
							if(c != null) c.Font = font;
							else { Debug.Assert(false); }
						};
						Action<Control> fM = delegate(Control c)
						{
							if(c != null) fC(c.ContextMenuStrip);
							else { Debug.Assert(false); }
						};

						fC(mf.MainMenu);
						fC(FindControl<ToolStrip>(mf, "m_toolMain"));
						fC(mf.GroupContextMenu);
						fC(mf.EntryContextMenu);
						fM(FindControl<RichTextBox>(mf, "m_richEntryView"));
						fC(FindControl<StatusStrip>(mf, "m_statusMain"));
						fC(mf.TrayContextMenu);

						fi = typeof(MainForm).GetField("m_tbQuickFind",
							BindingFlags.NonPublic | BindingFlags.Instance);
						if(fi != null)
						{
							ToolStripItem tsi = (fi.GetValue(mf) as ToolStripItem);
							if(tsi != null) tsi.Font = font;
							else { Debug.Assert(false); }
						}
						else { Debug.Assert(false); }
					}

					if(f.FormBorderStyle == FormBorderStyle.Sizable)
						f.Bounds = rOrg; // Restore
					else if(f.WindowState == FormWindowState.Normal)
					{
						Size szNew = f.Size;
						f.Location = new Point(rOrg.X - ((szNew.Width - rOrg.Width) / 2),
							rOrg.Y - ((szNew.Height - rOrg.Height) / 2));
					}
				}
			}
			catch(Exception) { Debug.Assert(false); }
		}

		private void CustomizeControlOnShown(Control c)
		{
			if(c == null) { Debug.Assert(false); return; }

			if(m_opt.CtrlBackspace)
			{
				GFunc<AutoCompleteStringCollection> fAcsc = delegate()
				{
					AutoCompleteStringCollection sc = new AutoCompleteStringCollection();
					sc.Add("-");
					return sc;
				};

				TextBox tb = (c as TextBox);
				if((tb != null) && (tb.AutoCompleteMode == AutoCompleteMode.None))
				{
					tb.AutoCompleteCustomSource = fAcsc();
					tb.AutoCompleteSource = AutoCompleteSource.CustomSource;
					tb.AutoCompleteMode = AutoCompleteMode.Suggest;
				}

				ComboBox cmb = (c as ComboBox);
				if((cmb != null) && (cmb.AutoCompleteMode == AutoCompleteMode.None) &&
					(cmb.DropDownStyle != ComboBoxStyle.DropDownList))
				{
					cmb.AutoCompleteCustomSource = fAcsc();
					cmb.AutoCompleteSource = AutoCompleteSource.CustomSource;
					cmb.AutoCompleteMode = AutoCompleteMode.Suggest;
				}
			}

			foreach(Control cSub in c.Controls)
				CustomizeControlOnShown(cSub);
		}

		private void OnFileOpened(object sender, FileOpenedEventArgs e)
		{
			PwDatabase pd = ((e != null) ? e.Database : null);
			if((pd == null) || !pd.IsOpen) { Debug.Assert(false); return; }

			bool bActive = (pd == m_host.Database);

			if(m_opt.MainGroupCollapseOnOpen)
			{
				PwGroup pgRoot = pd.RootGroup;
				if(pgRoot == null) { Debug.Assert(false); return; }

				pgRoot.TraverseTree(TraversalMethod.PreOrder, delegate(PwGroup pg)
				{
					pg.IsExpanded = false;
					return true;
				}, null);

				if(bActive)
				{
					m_host.MainWindow.UpdateUI(false, null, true, pgRoot, true, null, false);

					TreeView tv = FindControl<TreeView>(m_host.MainWindow, "m_tvGroups");
					if((tv != null) && (tv.Nodes.Count != 0))
						tv.Nodes[0].EnsureVisible();
					else { Debug.Assert(false); }
				}
			}

			if(m_opt.MainEntryFocusOnOpen && bActive)
			{
				ListView lv = FindEntryListView();
				if(lv != null)
				{
					Thread th = new Thread(delegate()
					{
						try
						{
							DelayAndFocus(0, lv);
							DelayAndFocus(50, lv);
							DelayAndFocus(200, lv);
							DelayAndFocus(250, lv);
							DelayAndFocus(500, lv);
						}
						catch(Exception) { Debug.Assert(false); }
					});
					th.Start();
				}
			}
		}

		private void DelayAndFocus(int ms, ListView lv)
		{
			Thread.Sleep(ms);

			lv.Invoke(new VoidDelegate(delegate()
			{
				try
				{
					if(!lv.Visible || !lv.Enabled) return;

					if((lv.Items.Count != 0) && (lv.SelectedIndices.Count == 0))
					{
						ListViewItem lvi = lv.Items[0];
						UIUtil.SetFocusedItem(lv, lvi, true);
						lvi.EnsureVisible();
					}

					m_host.MainWindow.ResetDefaultFocus(lv);
				}
				catch(Exception) { Debug.Assert(false); }
			}));
		}

		private void OnFileClosingPre(object sender, FileClosingEventArgs e)
		{
			if(e == null) { Debug.Assert(false); return; }
			if(!m_opt.ConfirmDatabaseClosing) return;

			string strText = KPRes.DatabaseFile + ":" + MessageService.NewLine +
				e.Database.IOConnectionInfo.GetDisplayName() + MessageService.NewParagraph +
				"Are you sure you want to close the database?";

			VistaTaskDialog vtd = new VistaTaskDialog();
			vtd.CommandLinks = false;
			vtd.Content = strText;
			vtd.MainInstruction = "Close database?";
			vtd.SetIcon(VtdCustomIcon.Question);
			vtd.VerificationText = KPRes.DialogNoShowAgain;
			vtd.WindowTitle = KeeUIExtExt.ProductName;

			vtd.AddButton((int)DialogResult.OK, KPRes.Yes, null);
			vtd.AddButton((int)DialogResult.Cancel, KPRes.No, null);

			if(vtd.ShowDialog())
			{
				if(vtd.Result == (int)DialogResult.OK)
				{
					if(vtd.ResultVerificationChecked)
						m_opt.ConfirmDatabaseClosing = false;
				}
				else e.Cancel = true;
			}
			else if(!MessageService.AskYesNo(strText, KeeUIExtExt.ProductName))
				e.Cancel = true;
		}
	}
}
