﻿/*
  KeeUIExt
  Copyright (C) 2014-2025 Dominik Reichl <dominik.reichl@t-online.de>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Text;

using KeePass.App.Configuration;

using KeePassLib.Delegates;
using KeePassLib.Utility;

namespace KeeUIExt
{
	internal sealed class KueOptions
	{
		private const string CfgFlags = "KeeUIExt_Flags";
		private const string CfgUIFont = "KeeUIExt_UIFont";

		// Obsolete item
		private const string CfgConfirmDatabaseClosing = "KeeUIExt_ConfirmDbClosing";

		public bool CtrlBackspace { get; set; }
		public bool ScrollMenusWithMouseWheel { get; set; }
		public bool ForgetLastDatabaseIfNotExists { get; set; }
		public bool ConfirmDatabaseClosing { get; set; }
		public bool MainGroupCollapseOnOpen { get; set; }
		public bool MainGroupContextDuplicate { get; set; }
		public bool MainEntryFocusOnOpen { get; set; }
		public bool MainEntryGridLines { get; set; }
		public bool MainEntryContextDuplicate { get; set; }
		public bool MainEntryContextPrint { get; set; }

		public AceFont UIFont { get; set; }

		public KueOptions() { }

		public static void Save(AceCustomConfig c, KueOptions o)
		{
			if((c == null) || (o == null)) { Debug.Assert(false); return; }

			c.SetString(CfgConfirmDatabaseClosing, null);

			ulong uFlags = 0, uFlag = 1;
			Action<bool> f = (b => { if(b) uFlags |= uFlag; uFlag <<= 1; });
			// New flags must be added at the end; do not remove any call
			f(o.ConfirmDatabaseClosing);
			f(o.CtrlBackspace);
			f(o.MainEntryGridLines);
			f(o.MainGroupCollapseOnOpen);
			f(o.MainEntryFocusOnOpen);
			f(o.MainEntryContextPrint);
			f(o.ForgetLastDatabaseIfNotExists);
			f(o.MainGroupContextDuplicate);
			f(o.MainEntryContextDuplicate);
			f(o.ScrollMenusWithMouseWheel);
			// Add new flags here

			if(uFlags == 0) c.SetString(CfgFlags, null);
			else c.SetULong(CfgFlags, uFlags);

			c.SetString(CfgUIFont, Serialize<AceFont>(o.UIFont));
		}

		public static KueOptions Load(AceCustomConfig c)
		{
			if(c == null) { Debug.Assert(false); return new KueOptions(); }

			KueOptions o = new KueOptions();

			ulong uFlags = c.GetULong(CfgFlags, 0), uFlag = 1;
			GFunc<bool> f = (() =>
			{
				bool b = ((uFlags & uFlag) != 0);
				uFlag <<= 1;
				return b;
			});
			// New flags must be added at the end; do not remove any call
			o.ConfirmDatabaseClosing = f();
			o.CtrlBackspace = f();
			o.MainEntryGridLines = f();
			o.MainGroupCollapseOnOpen = f();
			o.MainEntryFocusOnOpen = f();
			o.MainEntryContextPrint = f();
			o.ForgetLastDatabaseIfNotExists = f();
			o.MainGroupContextDuplicate = f();
			o.MainEntryContextDuplicate = f();
			o.ScrollMenusWithMouseWheel = f();
			// Add new flags here

			o.UIFont = Deserialize<AceFont>(c.GetString(CfgUIFont));

			return o;
		}

		private static string Serialize<T>(T t)
			where T : class
		{
			if(t == null) return null;

			using(MemoryStream ms = new MemoryStream())
			{
				XmlUtilEx.Serialize<T>(ms, t);
				return Convert.ToBase64String(ms.ToArray());
			}
		}

		private static T Deserialize<T>(string str)
			where T : class
		{
			if(str == null) return null;

			try
			{
				using(MemoryStream ms = new MemoryStream(
					Convert.FromBase64String(str), false))
				{
					return XmlUtilEx.Deserialize<T>(ms);
				}
			}
			catch(Exception) { Debug.Assert(false); }

			return null;
		}
	}
}
